// Service Worker Registration - Registro mejorado con manejo de actualizaciones

if ('serviceWorker' in navigator) {
    let registration = null;

    // Registrar Service Worker
    window.addEventListener('load', function() {
        registerServiceWorker();
    });

    async function registerServiceWorker() {
        try {
            // Obtener la ruta base dinámicamente
            const basePath = window.location.pathname.split('/vendedor')[0] || '';
            const swPath = basePath + '/vendedor/sw.js';
            const swScope = basePath + '/vendedor/';
            
            registration = await navigator.serviceWorker.register(swPath, {
                scope: swScope
            });

            console.log('[SW Register] Service Worker registrado:', registration.scope);

            // Verificar actualizaciones
            checkForUpdates();

            // Escuchar actualizaciones
            registration.addEventListener('updatefound', () => {
                console.log('[SW Register] Nueva versión del Service Worker encontrada');
                const newWorker = registration.installing;

                newWorker.addEventListener('statechange', () => {
                    if (newWorker.state === 'installed' && navigator.serviceWorker.controller) {
                        // Nueva versión disponible
                        showUpdateNotification();
                    }
                });
            });

            // Manejar mensajes del Service Worker
            navigator.serviceWorker.addEventListener('message', (event) => {
                console.log('[SW Register] Mensaje del Service Worker:', event.data);
                handleServiceWorkerMessage(event.data);
            });

        } catch (error) {
            console.error('[SW Register] Error registrando Service Worker:', error);
        }
    }

    function checkForUpdates() {
        if (registration) {
            registration.update().then(() => {
                console.log('[SW Register] Verificación de actualizaciones completada');
            }).catch(error => {
                console.error('[SW Register] Error verificando actualizaciones:', error);
            });
        }
    }

    function showUpdateNotification() {
        const notification = document.createElement('div');
        notification.id = 'sw-update-notification';
        notification.className = 'sw-update-notification';
        notification.innerHTML = `
            <div class="sw-update-content">
                <i class="fas fa-sync-alt"></i>
                <span>Nueva versión disponible</span>
                <button class="btn btn-primary btn-sm" onclick="updateServiceWorker()">
                    Actualizar
                </button>
            </div>
        `;
        document.body.appendChild(notification);
    }

    window.updateServiceWorker = function() {
        if (registration && registration.waiting) {
            registration.waiting.postMessage({ type: 'SKIP_WAITING' });
            window.location.reload();
        }
    };

    function handleServiceWorkerMessage(data) {
        switch (data.type) {
            case 'SYNC_COMPLETE':
                console.log('[SW Register] Sincronización completada');
                if (window.offlineDocumentManager) {
                    window.offlineDocumentManager.updateUI();
                }
                break;
            case 'SYNC_FAILED':
                console.error('[SW Register] Error en sincronización:', data.error);
                break;
            default:
                console.log('[SW Register] Mensaje desconocido:', data);
        }
    }

    // Verificar actualizaciones periódicamente
    setInterval(() => {
        checkForUpdates();
    }, 60 * 60 * 1000); // Cada hora
}
