// Push Notifications Manager - Gestión completa de notificaciones push
// Maneja la suscripción, desuscripción y recepción de notificaciones

class PushNotificationManager {
    constructor() {
        this.registration = null;
        this.subscription = null;
        this.publicKey = null;
        this.init();
    }

    async init() {
        if (!('serviceWorker' in navigator) || !('PushManager' in window)) {
            console.log('[PushManager] Push notifications no soportadas en este navegador');
            return;
        }

        try {
            // Obtener el Service Worker registration
            this.registration = await navigator.serviceWorker.ready;

            // Obtener la clave pública VAPID (no crítico si falla)
            await this.getPublicKey();

            // Solo continuar si tenemos la clave pública
            if (!this.publicKey) {
                console.log('[PushManager] Push notifications no configuradas (sin VAPID key) - Funcionalidad opcional');
                return;
            }

            // Verificar suscripción existente
            await this.checkSubscription();

            // Solicitar permiso si no está suscrito
            if (!this.subscription) {
                this.setupPermissionRequest();
            }
        } catch (error) {
            console.warn('[PushManager] Error inicializando (no crítico):', error);
            // No mostrar error al usuario, push notifications son opcionales
        }
    }

    async getPublicKey() {
        try {
            // Obtener la ruta base dinámicamente
            const basePath = window.location.pathname.split('/vendedor')[0] || '';
            const apiPath = basePath + '/api/push/public-key';
            
            const response = await fetch(apiPath, {
                method: 'GET',
                headers: {
                    'Accept': 'application/json',
                    'Content-Type': 'application/json'
                },
                credentials: 'same-origin'
            });
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const contentType = response.headers.get('content-type');
            if (!contentType || !contentType.includes('application/json')) {
                const text = await response.text();
                console.error('[PushManager] Respuesta no es JSON:', text.substring(0, 100));
                throw new Error('La respuesta del servidor no es JSON');
            }
            
            const data = await response.json();
            
            if (data.error) {
                console.warn('[PushManager] Error del servidor:', data.error);
                this.publicKey = null;
                return;
            }
            
            if (!data.publicKey) {
                console.warn('[PushManager] No se recibió la clave pública');
                this.publicKey = null;
                return;
            }
            
            this.publicKey = data.publicKey;
            console.log('[PushManager] Clave pública obtenida correctamente');
        } catch (error) {
            console.error('[PushManager] Error obteniendo clave pública:', error);
            this.publicKey = null;
            // No mostrar error al usuario si no hay clave configurada
            if (error.message.includes('404')) {
                console.warn('[PushManager] Endpoint de push notifications no disponible');
            }
        }
    }

    async checkSubscription() {
        try {
            this.subscription = await this.registration.pushManager.getSubscription();
            
            if (this.subscription) {
                console.log('[PushManager] Suscripción activa');
                // Verificar con el servidor
                await this.verifySubscription();
                this.updateUI(true);
            } else {
                console.log('[PushManager] No hay suscripción activa');
                this.updateUI(false);
            }
        } catch (error) {
            console.error('[PushManager] Error verificando suscripción:', error);
            this.updateUI(false);
        }
    }

    async subscribe() {
        if (!this.publicKey) {
            await this.getPublicKey();
        }

        if (!this.publicKey) {
            console.warn('[PushManager] No se pudo obtener la clave pública - Push notifications no disponibles');
            // No mostrar error al usuario si no hay clave configurada (es opcional)
            return false;
        }

        try {
            // Convertir la clave pública a formato Uint8Array
            const applicationServerKey = this.urlBase64ToUint8Array(this.publicKey);

            // Suscribirse
            this.subscription = await this.registration.pushManager.subscribe({
                userVisibleOnly: true,
                applicationServerKey: applicationServerKey
            });

            console.log('[PushManager] Suscripción creada:', this.subscription);

            // Enviar suscripción al servidor
            const success = await this.sendSubscriptionToServer();
            
            if (success) {
                this.updateUI(true);
                this.showSuccess('Notificaciones activadas correctamente');
                return true;
            } else {
                // Si falla el servidor, desuscribirse
                await this.unsubscribe();
                return false;
            }
        } catch (error) {
            console.error('[PushManager] Error suscribiéndose:', error);
            
            if (error.name === 'NotAllowedError') {
                this.showError('Permiso denegado para notificaciones');
            } else {
                this.showError('Error al activar notificaciones: ' + error.message);
            }
            return false;
        }
    }

    async unsubscribe() {
        try {
            if (this.subscription) {
                await this.subscription.unsubscribe();
                console.log('[PushManager] Desuscrito de push notifications');
            }

            // Notificar al servidor
            await this.removeSubscriptionFromServer();

            this.subscription = null;
            this.updateUI(false);
            this.showSuccess('Notificaciones desactivadas');
            return true;
        } catch (error) {
            console.error('[PushManager] Error desuscribiéndose:', error);
            this.showError('Error al desactivar notificaciones');
            return false;
        }
    }

    async verifySubscription() {
        if (!this.subscription) {
            return false;
        }

        try {
            const endpoint = this.subscription.endpoint;
            const basePath = window.location.pathname.split('/vendedor')[0] || '';
            const response = await fetch(basePath + '/api/push/verify-subscription', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
                },
                body: JSON.stringify({ endpoint })
            });

            const data = await response.json();
            if (!data.valid) {
                // La suscripción no es válida en el servidor, crear una nueva
                console.log('[PushManager] Suscripción no válida, recreando...');
                await this.subscribe();
            }
            return data.valid;
        } catch (error) {
            console.error('[PushManager] Error verificando suscripción:', error);
            return false;
        }
    }

    async sendSubscriptionToServer() {
        if (!this.subscription) {
            return false;
        }

        try {
            const subscriptionData = {
                endpoint: this.subscription.endpoint,
                keys: {
                    p256dh: this.arrayBufferToBase64(this.subscription.getKey('p256dh')),
                    auth: this.arrayBufferToBase64(this.subscription.getKey('auth'))
                }
            };

            const basePath = window.location.pathname.split('/vendedor')[0] || '';
            const response = await fetch(basePath + '/api/push/subscribe', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
                    'Accept': 'application/json'
                },
                body: JSON.stringify(subscriptionData)
            });

            if (response.ok) {
                console.log('[PushManager] Suscripción guardada en el servidor');
                return true;
            } else {
                console.error('[PushManager] Error guardando suscripción:', response.statusText);
                return false;
            }
        } catch (error) {
            console.error('[PushManager] Error enviando suscripción:', error);
            return false;
        }
    }

    async removeSubscriptionFromServer() {
        if (!this.subscription) {
            return;
        }

        try {
            const basePath = window.location.pathname.split('/vendedor')[0] || '';
            await fetch(basePath + '/api/push/unsubscribe', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
                    'Accept': 'application/json'
                },
                body: JSON.stringify({
                    endpoint: this.subscription.endpoint
                })
            });
        } catch (error) {
            console.error('[PushManager] Error removiendo suscripción:', error);
        }
    }

    setupPermissionRequest() {
        // Mostrar botón para activar notificaciones si el permiso no está denegado
        if (Notification.permission === 'default') {
            this.showPermissionBanner();
        }
    }

    showPermissionBanner() {
        const banner = document.createElement('div');
        banner.id = 'push-permission-banner';
        banner.className = 'push-permission-banner';
        banner.innerHTML = `
            <div class="push-banner-content">
                <i class="fas fa-bell"></i>
                <div class="push-banner-text">
                    <strong>Activa las notificaciones</strong>
                    <p>Recibe alertas importantes sobre tus clientes y documentos</p>
                </div>
                <button class="btn btn-primary" onclick="window.pushNotificationManager.subscribe()">
                    Activar
                </button>
                <button class="btn btn-secondary" onclick="this.closest('.push-permission-banner').remove()">
                    Ahora no
                </button>
            </div>
        `;
        document.body.appendChild(banner);
    }

    updateUI(isSubscribed) {
        const toggle = document.getElementById('push-notification-toggle');
        if (toggle) {
            toggle.checked = isSubscribed;
            toggle.onchange = () => {
                if (toggle.checked) {
                    this.subscribe();
                } else {
                    this.unsubscribe();
                }
            };
        }
    }

    urlBase64ToUint8Array(base64String) {
        const padding = '='.repeat((4 - base64String.length % 4) % 4);
        const base64 = (base64String + padding)
            .replace(/\-/g, '+')
            .replace(/_/g, '/');

        const rawData = window.atob(base64);
        const outputArray = new Uint8Array(rawData.length);

        for (let i = 0; i < rawData.length; ++i) {
            outputArray[i] = rawData.charCodeAt(i);
        }
        return outputArray;
    }

    arrayBufferToBase64(buffer) {
        const bytes = new Uint8Array(buffer);
        let binary = '';
        for (let i = 0; i < bytes.byteLength; i++) {
            binary += String.fromCharCode(bytes[i]);
        }
        return window.btoa(binary);
    }

    showSuccess(message) {
        this.showNotification(message, 'success');
    }

    showError(message) {
        this.showNotification(message, 'error');
    }

    showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `toast-notification ${type}`;
        notification.innerHTML = `
            <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
            <span>${message}</span>
        `;
        document.body.appendChild(notification);

        setTimeout(() => {
            notification.classList.add('show');
        }, 100);

        setTimeout(() => {
            notification.classList.remove('show');
            setTimeout(() => {
                notification.remove();
            }, 300);
        }, 3000);
    }
}

// Inicializar cuando el Service Worker esté listo
if ('serviceWorker' in navigator) {
    navigator.serviceWorker.ready.then(() => {
        window.pushNotificationManager = new PushNotificationManager();
    });
}
