// Offline Document Manager - Gestión completa de documentos offline
// Maneja la cola de documentos pendientes usando IndexedDB

const OFFLINE_QUEUE_DB = 'offline_queue_db';
const OFFLINE_QUEUE_STORE = 'documents';
const DB_VERSION = 1;

class OfflineDocumentManager {
    constructor() {
        this.db = null;
        this.init();
        this.setupEventListeners();
        this.updateUI();
    }

    async init() {
        try {
            this.db = await this.openDB();
            console.log('[OfflineManager] IndexedDB inicializado');
            this.updateUI();
        } catch (error) {
            console.error('[OfflineManager] Error inicializando IndexedDB:', error);
        }
    }

    openDB() {
        return new Promise((resolve, reject) => {
            const request = indexedDB.open(OFFLINE_QUEUE_DB, DB_VERSION);

            request.onerror = () => reject(request.error);
            request.onsuccess = () => resolve(request.result);

            request.onupgradeneeded = (event) => {
                const db = event.target.result;
                if (!db.objectStoreNames.contains(OFFLINE_QUEUE_STORE)) {
                    const store = db.createObjectStore(OFFLINE_QUEUE_STORE, {
                        keyPath: 'id',
                        autoIncrement: true
                    });
                    store.createIndex('timestamp', 'timestamp', { unique: false });
                    store.createIndex('status', 'status', { unique: false });
                }
            };
        });
    }

    setupEventListeners() {
        // Detectar cambios de conexión
        window.addEventListener('online', () => {
            console.log('[OfflineManager] Conexión restaurada');
            this.updateUI();
            this.syncPendingDocuments();
        });

        window.addEventListener('offline', () => {
            console.log('[OfflineManager] Sin conexión');
            this.updateUI();
        });

        // Sincronizar periódicamente cuando hay conexión
        if (navigator.onLine) {
            setInterval(() => {
                if (navigator.onLine) {
                    this.syncPendingDocuments();
                }
            }, 30000); // Cada 30 segundos
        }
    }

    async addToQueue(formData, url) {
        if (!this.db) {
            await this.init();
        }

        try {
            // Serializar FormData (esperar a que termine)
            const serializedData = await this.serializeFormData(formData);

            return new Promise((resolve, reject) => {
                const transaction = this.db.transaction([OFFLINE_QUEUE_STORE], 'readwrite');
                const store = transaction.objectStore(OFFLINE_QUEUE_STORE);

                const item = {
                    url: url,
                    method: 'POST',
                    formData: serializedData,
                    timestamp: Date.now(),
                    status: 'pending',
                    retries: 0
                };

                const request = store.add(item);

                request.onsuccess = () => {
                    console.log('[OfflineManager] Documento agregado a la cola:', request.result);
                    this.updateUI();
                    this.showNotification('Documento guardado para sincronizar cuando vuelva la conexión', 'info');
                    resolve(request.result);
                };

                request.onerror = () => {
                    console.error('[OfflineManager] Error agregando a la cola:', request.error);
                    reject(request.error);
                };
            });
        } catch (error) {
            console.error('[OfflineManager] Error serializando FormData:', error);
            throw error;
        }
    }

    async serializeFormData(formData) {
        const data = {};
        const promises = [];
        
        for (const [key, value] of formData.entries()) {
            if (value instanceof File) {
                // Convertir File a objeto serializable de forma asíncrona
                const filePromise = new Promise((resolve) => {
                    const reader = new FileReader();
                    reader.onload = (e) => {
                        data[key] = {
                            type: 'file',
                            name: value.name,
                            type_mime: value.type,
                            size: value.size,
                            lastModified: value.lastModified,
                            data: Array.from(new Uint8Array(e.target.result))
                        };
                        resolve();
                    };
                    reader.onerror = () => {
                        console.error('[OfflineManager] Error leyendo archivo:', value.name);
                        resolve();
                    };
                    reader.readAsArrayBuffer(value);
                });
                promises.push(filePromise);
            } else {
                data[key] = value;
            }
        }
        
        // Esperar a que todos los archivos se lean
        await Promise.all(promises);
        return data;
    }

    async getPendingCount() {
        if (!this.db) {
            await this.init();
        }

        return new Promise((resolve) => {
            const transaction = this.db.transaction([OFFLINE_QUEUE_STORE], 'readonly');
            const store = transaction.objectStore(OFFLINE_QUEUE_STORE);
            const index = store.index('status');
            const request = index.count(IDBKeyRange.only('pending'));

            request.onsuccess = () => {
                resolve(request.result || 0);
            };

            request.onerror = () => {
                resolve(0);
            };
        });
    }

    async getAllPending() {
        if (!this.db) {
            await this.init();
        }

        return new Promise((resolve) => {
            const transaction = this.db.transaction([OFFLINE_QUEUE_STORE], 'readonly');
            const store = transaction.objectStore(OFFLINE_QUEUE_STORE);
            const index = store.index('status');
            const request = index.getAll(IDBKeyRange.only('pending'));

            request.onsuccess = () => {
                resolve(request.result || []);
            };

            request.onerror = () => {
                resolve([]);
            };
        });
    }

    async syncPendingDocuments() {
        if (!navigator.onLine) {
            console.log('[OfflineManager] Sin conexión, no se puede sincronizar');
            return;
        }

        const pending = await this.getAllPending();
        if (pending.length === 0) {
            this.updateUI();
            return;
        }

        console.log(`[OfflineManager] Sincronizando ${pending.length} documentos...`);
        this.showSyncProgress(pending.length);

        let successCount = 0;
        let errorCount = 0;

        for (const item of pending) {
            try {
                // Reconstruir FormData
                const formData = new FormData();
                for (const [key, value] of Object.entries(item.formData)) {
                    if (value && value.type === 'file') {
                        const blob = new Blob([new Uint8Array(value.data)], {
                            type: value.type_mime
                        });
                        const file = new File([blob], value.name, {
                            type: value.type_mime,
                            lastModified: value.lastModified
                        });
                        formData.append(key, file);
                    } else {
                        formData.append(key, value);
                    }
                }

                const response = await fetch(item.url, {
                    method: item.method,
                    body: formData,
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest'
                    }
                });

                if (response.ok) {
                    // Eliminar de la cola
                    await this.removeFromQueue(item.id);
                    successCount++;
                    console.log(`[OfflineManager] Documento ${item.id} sincronizado exitosamente`);
                } else {
                    errorCount++;
                    item.retries++;
                    if (item.retries >= 3) {
                        await this.markAsFailed(item.id);
                    } else {
                        await this.updateItem(item);
                    }
                }
            } catch (error) {
                console.error(`[OfflineManager] Error sincronizando documento ${item.id}:`, error);
                errorCount++;
                item.retries++;
                if (item.retries >= 3) {
                    await this.markAsFailed(item.id);
                } else {
                    await this.updateItem(item);
                }
            }
        }

        this.hideSyncProgress();
        this.updateUI();

        if (successCount > 0) {
            this.showNotification(`${successCount} documento(s) sincronizado(s) exitosamente`, 'success');
        }
        if (errorCount > 0) {
            this.showNotification(`${errorCount} documento(s) con error. Reintentando...`, 'warning');
        }
    }

    async removeFromQueue(id) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([OFFLINE_QUEUE_STORE], 'readwrite');
            const store = transaction.objectStore(OFFLINE_QUEUE_STORE);
            const request = store.delete(id);

            request.onsuccess = () => resolve();
            request.onerror = () => reject(request.error);
        });
    }

    async markAsFailed(id) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([OFFLINE_QUEUE_STORE], 'readwrite');
            const store = transaction.objectStore(OFFLINE_QUEUE_STORE);
            const request = store.get(id);

            request.onsuccess = () => {
                const item = request.result;
                if (item) {
                    item.status = 'failed';
                    const updateRequest = store.put(item);
                    updateRequest.onsuccess = () => resolve();
                    updateRequest.onerror = () => reject(updateRequest.error);
                } else {
                    resolve();
                }
            };

            request.onerror = () => reject(request.error);
        });
    }

    async updateItem(item) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([OFFLINE_QUEUE_STORE], 'readwrite');
            const store = transaction.objectStore(OFFLINE_QUEUE_STORE);
            const request = store.put(item);

            request.onsuccess = () => resolve();
            request.onerror = () => reject(request.error);
        });
    }

    updateUI() {
        this.getPendingCount().then(count => {
            // Actualizar badge de documentos pendientes
            const badge = document.getElementById('offline-docs-badge');
            if (badge) {
                if (count > 0) {
                    badge.textContent = count;
                    badge.style.display = 'inline-block';
                } else {
                    badge.style.display = 'none';
                }
            }

            // Actualizar indicador de estado
            const statusIndicator = document.getElementById('connection-status');
            if (statusIndicator) {
                if (navigator.onLine) {
                    statusIndicator.className = 'connection-status online';
                    statusIndicator.innerHTML = '<i class="fas fa-wifi"></i> En línea';
                } else {
                    statusIndicator.className = 'connection-status offline';
                    statusIndicator.innerHTML = '<i class="fas fa-wifi-slash"></i> Sin conexión';
                }
            }
        });
    }

    showSyncProgress(total) {
        let progressBar = document.getElementById('sync-progress');
        if (!progressBar) {
            progressBar = document.createElement('div');
            progressBar.id = 'sync-progress';
            progressBar.className = 'sync-progress';
            progressBar.innerHTML = `
                <div class="sync-progress-content">
                    <i class="fas fa-sync fa-spin"></i>
                    <span>Sincronizando documentos...</span>
                </div>
            `;
            document.body.appendChild(progressBar);
        }
        progressBar.style.display = 'flex';
    }

    hideSyncProgress() {
        const progressBar = document.getElementById('sync-progress');
        if (progressBar) {
            progressBar.style.display = 'none';
        }
    }

    showNotification(message, type = 'info') {
        // Crear notificación toast
        const notification = document.createElement('div');
        notification.className = `toast-notification ${type}`;
        notification.innerHTML = `
            <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'warning' ? 'exclamation-triangle' : 'info-circle'}"></i>
            <span>${message}</span>
        `;
        document.body.appendChild(notification);

        setTimeout(() => {
            notification.classList.add('show');
        }, 100);

        setTimeout(() => {
            notification.classList.remove('show');
            setTimeout(() => {
                notification.remove();
            }, 300);
        }, 3000);
    }
}

// Interceptar envío de formularios de documentos
document.addEventListener('DOMContentLoaded', function() {
    window.offlineDocumentManager = new OfflineDocumentManager();

    // Interceptar formularios de documentos
    const documentForms = document.querySelectorAll('form[action*="documentos"]');
    documentForms.forEach(form => {
        form.addEventListener('submit', async function(e) {
            if (!navigator.onLine) {
                e.preventDefault();
                const formData = new FormData(form);
                const action = form.getAttribute('action');
                await window.offlineDocumentManager.addToQueue(formData, action);
                return false;
            }
        });
    });

    // Botón de sincronización manual
    const syncBtn = document.getElementById('sync-pending-btn');
    if (syncBtn) {
        syncBtn.addEventListener('click', () => {
            window.offlineDocumentManager.syncPendingDocuments();
        });
    }
});
