/**
 * Sistema de Modales para Confirmaciones
 */

class ModalManager {
    constructor() {
        this.modalContainer = null;
        this.init();
    }

    init() {
        // Crear contenedor de modal si no existe
        if (!document.getElementById('modal-container')) {
            this.modalContainer = document.createElement('div');
            this.modalContainer.id = 'modal-container';
            this.modalContainer.className = 'modal-overlay';
            this.modalContainer.innerHTML = `
                <div class="modal-content">
                    <div class="modal-header">
                        <h3 class="modal-title" id="modal-title"></h3>
                        <button type="button" class="modal-close" onclick="modalManager.close()">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                    <div class="modal-body" id="modal-body"></div>
                    <div class="modal-footer" id="modal-footer"></div>
                </div>
            `;
            document.body.appendChild(this.modalContainer);
        } else {
            this.modalContainer = document.getElementById('modal-container');
        }

        // Cerrar al hacer click fuera del modal
        this.modalContainer.addEventListener('click', (e) => {
            if (e.target === this.modalContainer) {
                this.close();
            }
        });

        // Cerrar con ESC
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape' && this.modalContainer.classList.contains('active')) {
                this.close();
            }
        });
    }

    confirm(options) {
        return new Promise((resolve) => {
            const {
                title = 'Confirmar Acción',
                message = '¿Estás seguro de realizar esta acción?',
                confirmText = 'Confirmar',
                cancelText = 'Cancelar',
                confirmClass = 'btn-danger',
                icon = 'fas fa-exclamation-triangle'
            } = options;

            document.getElementById('modal-title').innerHTML = `
                <i class="${icon}"></i> ${title}
            `;
            document.getElementById('modal-body').innerHTML = `<p>${message}</p>`;
            document.getElementById('modal-footer').innerHTML = `
                <button type="button" class="btn btn-secondary" onclick="modalManager.close(); modalManager.resolve(false);">
                    <i class="fas fa-times"></i> ${cancelText}
                </button>
                <button type="button" class="btn ${confirmClass}" onclick="modalManager.close(); modalManager.resolve(true);">
                    <i class="fas fa-check"></i> ${confirmText}
                </button>
            `;

            this.modalContainer.classList.add('active');
            this.resolveCallback = resolve;
        });
    }

    resolve(value) {
        if (this.resolveCallback) {
            this.resolveCallback(value);
            this.resolveCallback = null;
        }
    }

    close() {
        this.modalContainer.classList.remove('active');
    }

    alert(options) {
        return new Promise((resolve) => {
            const {
                title = 'Información',
                message = '',
                buttonText = 'Aceptar',
                icon = 'fas fa-info-circle',
                type = 'info'
            } = options;

            const typeClass = {
                'info': 'btn-primary',
                'success': 'btn-success',
                'warning': 'btn-warning',
                'danger': 'btn-danger'
            }[type] || 'btn-primary';

            document.getElementById('modal-title').innerHTML = `
                <i class="${icon}"></i> ${title}
            `;
            document.getElementById('modal-body').innerHTML = `<p>${message}</p>`;
            document.getElementById('modal-footer').innerHTML = `
                <button type="button" class="btn ${typeClass}" onclick="modalManager.close(); modalManager.resolve(true);">
                    <i class="fas fa-check"></i> ${buttonText}
                </button>
            `;

            this.modalContainer.classList.add('active');
            this.resolveCallback = resolve;
        });
    }
}

// Instancia global
const modalManager = new ModalManager();

// Reemplazar confirm() nativo
window.confirmModal = function(message, options = {}) {
    return modalManager.confirm({
        message,
        ...options
    });
};

// Helper para formularios de eliminación
document.addEventListener('DOMContentLoaded', function() {
    // Reemplazar confirm() en formularios de eliminación
    document.querySelectorAll('form[onsubmit*="confirm"]').forEach(form => {
        const originalOnsubmit = form.onsubmit;
        form.onsubmit = null;
        
        form.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const confirmed = await modalManager.confirm({
                title: 'Confirmar Eliminación',
                message: '¿Estás seguro de eliminar este elemento? Esta acción no se puede deshacer.',
                confirmText: 'Eliminar',
                cancelText: 'Cancelar',
                confirmClass: 'btn-danger',
                icon: 'fas fa-trash-alt'
            });

            if (confirmed) {
                form.submit();
            }
        });
    });

    // Manejar formularios con clase delete-form
    document.querySelectorAll('form.delete-form').forEach(form => {
        form.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const confirmed = await modalManager.confirm({
                title: 'Confirmar Eliminación',
                message: '¿Estás seguro de eliminar este elemento? Esta acción no se puede deshacer.',
                confirmText: 'Eliminar',
                cancelText: 'Cancelar',
                confirmClass: 'btn-danger',
                icon: 'fas fa-trash-alt'
            });

            if (confirmed) {
                // Agregar loading state al botón
                const btn = form.querySelector('button[type="submit"]');
                if (btn) {
                    btn.classList.add('loading');
                    btn.disabled = true;
                }
                form.submit();
            }
        });
    });

    // Loading states para todos los botones de submit
    // IMPORTANTE: No prevenir el envío, solo agregar feedback visual
    document.querySelectorAll('form:not(.delete-form)').forEach(form => {
        form.addEventListener('submit', function(e) {
            // NO prevenir el envío (no usar e.preventDefault())
            const submitBtn = form.querySelector('button[type="submit"]');
            
            if (submitBtn && !submitBtn.disabled && !submitBtn.classList.contains('loading')) {
                // Verificar validación HTML5
                if (!form.checkValidity()) {
                    // Si no es válido, el navegador mostrará los mensajes
                    // No hacer nada más
                    return;
                }
                
                // Solo agregar loading si el formulario es válido
                // El formulario se enviará normalmente
                const originalText = submitBtn.innerHTML;
                
                // Agregar estado de loading visualmente
                submitBtn.classList.add('loading');
                submitBtn.disabled = true;
                
                // Cambiar texto para feedback visual
                if (!originalText.includes('fa-spinner')) {
                    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Guardando...';
                }
                
                // Fallback: Si después de 15 segundos el botón sigue deshabilitado, restaurarlo
                // (normalmente el formulario se envía y la página se recarga antes)
                const timeoutId = setTimeout(() => {
                    if (submitBtn && submitBtn.disabled && submitBtn.classList.contains('loading')) {
                        submitBtn.classList.remove('loading');
                        submitBtn.disabled = false;
                        submitBtn.innerHTML = originalText;
                    }
                }, 15000);
                
                // Limpiar timeout si el formulario se envía correctamente
                // (aunque normalmente la página se recarga)
                form.addEventListener('submit', () => clearTimeout(timeoutId), { once: true });
            }
        }, { passive: true }); // Marcar como pasivo para mejor rendimiento
    });
});

