<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class Documento extends Model
{
    use HasFactory;

    protected $fillable = [
        'cliente_id',
        'tipo_documento_id',
        'nombre_archivo',
        'ruta_archivo',
        'fecha_vencimiento',
        'observaciones',
        'creado_por_id',
    ];

    protected function casts(): array
    {
        return [
            'fecha_vencimiento' => 'date',
        ];
    }

    // Relaciones
    public function cliente(): BelongsTo
    {
        return $this->belongsTo(Cliente::class);
    }

    public function tipoDocumento(): BelongsTo
    {
        return $this->belongsTo(TipoDocumento::class, 'tipo_documento_id');
    }

    public function creadoPor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'creado_por_id');
    }

    // Scopes
    public function scopeVencidos($query)
    {
        return $query->whereNotNull('fecha_vencimiento')
            ->where('fecha_vencimiento', '<', now());
    }

    public function scopeProximosAVencer($query, $meses = 6)
    {
        return $query->whereNotNull('fecha_vencimiento')
            ->where('fecha_vencimiento', '>=', now())
            ->where('fecha_vencimiento', '<=', now()->addMonths($meses));
    }

    // Accessors
    public function getEstaVencidoAttribute(): bool
    {
        if (!$this->fecha_vencimiento) {
            return false;
        }
        return Carbon::parse($this->fecha_vencimiento)->isPast();
    }

    public function getProximoAVencerAttribute(): bool
    {
        if (!$this->fecha_vencimiento) {
            return false;
        }
        $fecha = Carbon::parse($this->fecha_vencimiento);
        return $fecha->isFuture() && $fecha->lte(now()->addMonths(6));
    }
}

