<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Cliente extends Model
{
    use HasFactory;

    protected $fillable = [
        'nombre_empresa',
        'rif',
        'direccion_fiscal',
        'zona',
        'vendedor_id',
        'notas',
        'creado_por_id',
        'modificado_por_id',
    ];

    // Relaciones
    public function vendedor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'vendedor_id');
    }

    public function creadoPor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'creado_por_id');
    }

    public function modificadoPor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'modificado_por_id');
    }

    public function documentos(): HasMany
    {
        return $this->hasMany(Documento::class);
    }

    public function solicitudesCreacion(): HasMany
    {
        return $this->hasMany(SolicitudCreacion::class);
    }

    public function solicitudesEdicion(): HasMany
    {
        return $this->hasMany(SolicitudEdicion::class);
    }

    public function historialCambios(): HasMany
    {
        return $this->hasMany(HistorialCambio::class);
    }

    // Scopes
    public function scopeDeVendedor($query, $vendedorId)
    {
        return $query->where('vendedor_id', $vendedorId);
    }

    // Métodos de negocio
    public function calcularEstatus(): array
    {
        $documentos = $this->documentos;
        
        if ($documentos->isEmpty()) {
            return [
                'texto' => 'SIN DOCUMENTOS',
                'clase' => 'sin-documentos',
                'icono' => 'fas fa-folder-minus'
            ];
        }

        $hoy = now()->startOfDay();
        $alertaVencido = false;
        $alertaProximoVencer = false;
        $tieneDocumentosConVencimiento = false;
        $fechaLimiteAlerta = now()->addMonths(6);

        foreach ($documentos as $doc) {
            if (!$doc->fecha_vencimiento) {
                continue;
            }

            $tieneDocumentosConVencimiento = true;
            $fechaVencimiento = \Carbon\Carbon::parse($doc->fecha_vencimiento)->startOfDay();

            if ($fechaVencimiento->lt($hoy)) {
                $alertaVencido = true;
                break;
            }

            if ($fechaVencimiento->lte($fechaLimiteAlerta)) {
                $alertaProximoVencer = true;
            }
        }

        if ($alertaVencido) {
            return [
                'texto' => 'VENCIDO',
                'clase' => 'vencido',
                'icono' => 'fas fa-times-circle'
            ];
        }

        if ($alertaProximoVencer) {
            return [
                'texto' => 'ALERTA',
                'clase' => 'alerta',
                'icono' => 'fas fa-exclamation-triangle'
            ];
        }

        return [
            'texto' => 'ACTUALIZADO',
            'clase' => 'actualizado',
            'icono' => 'fas fa-check-circle'
        ];
    }

    // Accessors
    public function getEstatusAttribute(): array
    {
        return $this->calcularEstatus();
    }
}

