<?php

namespace App\Http\Controllers\Vendedor;

use App\Http\Controllers\Controller;
use App\Models\SolicitudCreacion;
use App\Models\SolicitudEdicion;
use App\Models\Cliente;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SolicitudController extends Controller
{
    public function create()
    {
        return view('vendedor.solicitudes.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nombre_empresa' => 'required|string|max:255',
            'rif' => 'required|string|max:50',
            'direccion_fiscal' => 'nullable|string',
            'zona' => 'nullable|string|max:100',
            'notas' => 'nullable|string',
        ]);

        SolicitudCreacion::create([
            'vendedor_id' => Auth::id(),
            'nombre_empresa' => $validated['nombre_empresa'],
            'rif' => $validated['rif'],
            'direccion_fiscal' => $validated['direccion_fiscal'] ?? null,
            'zona' => $validated['zona'] ?? null,
            'notas' => $validated['notas'] ?? null,
            'estatus' => 'Pendiente',
        ]);

        return redirect()->route('vendedor.dashboard', ['view' => 'solicitudes'])
            ->with('success', 'Solicitud enviada exitosamente. Será revisada por un administrador.');
    }

    public function solicitarEdicion(Cliente $cliente)
    {
        // Verificar que el cliente pertenece al vendedor
        if ($cliente->vendedor_id !== Auth::id()) {
            abort(403, 'No tienes acceso a este cliente.');
        }

        return view('vendedor.solicitudes.editar-cliente', compact('cliente'));
    }

    public function storeEdicion(Request $request, Cliente $cliente)
    {
        // Verificar acceso
        if ($cliente->vendedor_id !== Auth::id()) {
            abort(403);
        }

        $validated = $request->validate([
            'nombre_empresa' => 'nullable|string|max:255',
            'rif' => 'nullable|string|max:50',
            'direccion_fiscal' => 'nullable|string',
            'zona' => 'nullable|string|max:100',
            'vendedor_id' => 'nullable|exists:users,id',
            'notas' => 'nullable|string',
        ]);

        // Filtrar solo los campos que cambiaron
        $cambios = [];
        foreach ($validated as $key => $value) {
            if ($value !== null && $cliente->$key != $value) {
                $cambios[$key] = $value;
            }
        }

        if (empty($cambios)) {
            return redirect()->back()
                ->withErrors(['error' => 'No hay cambios para solicitar.']);
        }

        SolicitudEdicion::create([
            'cliente_id' => $cliente->id,
            'vendedor_id' => Auth::id(),
            'cambios' => $cambios,
            'estatus' => 'Pendiente',
        ]);

        return redirect()->route('vendedor.clientes.show', $cliente)
            ->with('success', 'Solicitud de edición enviada exitosamente.');
    }
}

