<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;
use App\Models\AuditoriaAcceso;

class AuthController extends Controller
{
    public function showLoginForm()
    {
        if (Auth::check()) {
            $user = Auth::user();
            if ($user->isAdmin()) {
                return redirect()->route('admin.dashboard');
            }
        }
        return view('auth.login');
    }

    public function showVendedorLoginForm()
    {
        if (Auth::check()) {
            $user = Auth::user();
            if ($user->isVendedor()) {
                return redirect()->route('vendedor.dashboard');
            }
        }
        return view('vendedor.login');
    }

    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if (Auth::attempt($credentials, $request->boolean('remember'))) {
            $request->session()->regenerate();
            $user = Auth::user();

            if (!$user->activo) {
                Auth::logout();
                
                // Registrar intento fallido (cuenta desactivada)
                AuditoriaAcceso::create([
                    'email' => $credentials['email'],
                    'ip_address' => $request->ip(),
                    'user_agent' => $request->userAgent(),
                    'tipo' => 'login_fallido',
                    'detalles' => 'Cuenta desactivada',
                ]);
                
                throw ValidationException::withMessages([
                    'email' => 'Tu cuenta está desactivada.',
                ]);
            }

            // Registrar login exitoso
            AuditoriaAcceso::create([
                'usuario_id' => $user->id,
                'email' => $user->email,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'tipo' => 'login_exitoso',
            ]);

            if ($user->isAdmin()) {
                return redirect()->intended(route('admin.dashboard'));
            } elseif ($user->isVendedor()) {
                return redirect()->intended(route('vendedor.dashboard'));
            }
        }

        // Registrar intento fallido
        AuditoriaAcceso::create([
            'email' => $credentials['email'],
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'tipo' => 'login_fallido',
            'detalles' => 'Credenciales incorrectas',
        ]);

        throw ValidationException::withMessages([
            'email' => 'Las credenciales proporcionadas no son correctas.',
        ]);
    }

    public function vendedorLogin(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        // Verificar que el usuario es vendedor
        $user = \App\Models\User::where('email', $credentials['email'])
            ->whereHas('rol', function($q) {
                $q->where('nombre_rol', 'Vendedor');
            })
            ->first();

        if ($user && \Illuminate\Support\Facades\Hash::check($credentials['password'], $user->password)) {
            if (!$user->activo) {
                AuditoriaAcceso::create([
                    'email' => $credentials['email'],
                    'ip_address' => $request->ip(),
                    'user_agent' => $request->userAgent(),
                    'tipo' => 'login_fallido',
                    'detalles' => 'Cuenta desactivada',
                ]);
                
                throw ValidationException::withMessages([
                    'email' => 'Tu cuenta está desactivada.',
                ]);
            }

            Auth::login($user, $request->boolean('remember'));
            $request->session()->regenerate();

            // Registrar login exitoso
            AuditoriaAcceso::create([
                'usuario_id' => $user->id,
                'email' => $user->email,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'tipo' => 'login_exitoso',
            ]);

            return redirect()->intended(route('vendedor.dashboard'));
        }

        // Registrar intento fallido
        AuditoriaAcceso::create([
            'email' => $credentials['email'],
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'tipo' => 'login_fallido',
            'detalles' => 'Credenciales incorrectas o no es vendedor',
        ]);

        throw ValidationException::withMessages([
            'email' => 'Credenciales incorrectas o no tienes permiso de vendedor.',
        ]);
    }

    public function logout(Request $request)
    {
        try {
            $user = Auth::user();
            
            // Registrar logout
            if ($user) {
                AuditoriaAcceso::create([
                    'usuario_id' => $user->id,
                    'email' => $user->email,
                    'ip_address' => $request->ip(),
                    'user_agent' => $request->userAgent(),
                    'tipo' => 'logout',
                ]);
            }
        } catch (\Exception $e) {
            // Si falla la auditoría, continuar con el logout
            \Log::error('Error al registrar logout: ' . $e->getMessage());
        }
        
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        // Redirigir según el tipo de usuario
        $route = $request->is('vendedor/*') ? 'vendedor.login' : 'login';
        return redirect()->route($route);
    }
}

