<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SolicitudCreacion;
use App\Models\SolicitudEdicion;
use App\Models\Cliente;
use App\Models\HistorialCambio;
use App\Services\ClienteService;
use Illuminate\Http\Request;

class SolicitudController extends Controller
{
    protected $clienteService;

    public function __construct(ClienteService $clienteService)
    {
        $this->clienteService = $clienteService;
    }

    public function index(Request $request)
    {
        $perPage = $request->get('per_page', 20);
        $solicitudesCreacion = SolicitudCreacion::with(['vendedor', 'revisadoPor'])
            ->orderBy('fecha_solicitud', 'desc')
            ->paginate($perPage)
            ->withQueryString();

        $solicitudesEdicion = SolicitudEdicion::with(['cliente', 'vendedor', 'revisadoPor'])
            ->orderBy('fecha_solicitud', 'desc')
            ->paginate($perPage)
            ->withQueryString();

        return view('admin.solicitudes.index', [
            'solicitudesCreacion' => $solicitudesCreacion,
            'solicitudesEdicion' => $solicitudesEdicion,
        ]);
    }

    public function aprobarCreacion(SolicitudCreacion $solicitud)
    {
        if ($solicitud->estatus !== 'Pendiente') {
            return redirect()->back()
                ->withErrors(['error' => 'Esta solicitud ya fue procesada.']);
        }

        // Crear cliente
        $cliente = $this->clienteService->crearCliente([
            'nombre_empresa' => $solicitud->nombre_empresa,
            'rif' => $solicitud->rif,
            'direccion_fiscal' => $solicitud->direccion_fiscal,
            'zona' => $solicitud->zona,
            'vendedor_id' => $solicitud->vendedor_id,
            'notas' => $solicitud->notas,
        ], auth()->id());

        // Actualizar solicitud
        $solicitud->update([
            'estatus' => 'Aprobada',
            'revisado_por_id' => auth()->id(),
            'fecha_revision' => now(),
        ]);

        // Registrar en historial (ya se registra en crearCliente, pero agregamos nota de solicitud)
        HistorialCambio::create([
            'cliente_id' => $cliente->id,
            'usuario_id' => auth()->id(),
            'accion_realizada' => "Cliente creado desde solicitud de creación (ID: {$solicitud->id}).",
        ]);

        return redirect()->back()
            ->with('success', 'Solicitud aprobada y cliente creado exitosamente.');
    }

    public function rechazarCreacion(Request $request, SolicitudCreacion $solicitud)
    {
        $validated = $request->validate([
            'observaciones' => 'required|string',
        ]);

        $solicitud->update([
            'estatus' => 'Rechazada',
            'revisado_por_id' => auth()->id(),
            'fecha_revision' => now(),
            'observaciones' => $validated['observaciones'],
        ]);

        return redirect()->back()
            ->with('success', 'Solicitud rechazada.');
    }

    public function aprobarEdicion(SolicitudEdicion $solicitud)
    {
        if ($solicitud->estatus !== 'Pendiente') {
            return redirect()->back()
                ->withErrors(['error' => 'Esta solicitud ya fue procesada.']);
        }

        $cliente = $solicitud->cliente;
        $cambios = $solicitud->cambios;

        // Aplicar cambios al cliente
        $this->clienteService->actualizarCliente($cliente, $cambios, auth()->id());

        // Actualizar solicitud
        $solicitud->update([
            'estatus' => 'Aprobada',
            'revisado_por_id' => auth()->id(),
            'fecha_revision' => now(),
        ]);

        // Registrar en historial (ya se registra en actualizarCliente, pero agregamos nota de solicitud)
        HistorialCambio::create([
            'cliente_id' => $cliente->id,
            'usuario_id' => auth()->id(),
            'accion_realizada' => "Cambios aplicados desde solicitud de edición (ID: {$solicitud->id}).",
        ]);

        return redirect()->back()
            ->with('success', 'Solicitud aprobada y cambios aplicados.');
    }

    public function rechazarEdicion(Request $request, SolicitudEdicion $solicitud)
    {
        $validated = $request->validate([
            'observaciones' => 'required|string',
        ]);

        $solicitud->update([
            'estatus' => 'Rechazada',
            'revisado_por_id' => auth()->id(),
            'fecha_revision' => now(),
            'observaciones' => $validated['observaciones'],
        ]);

        return redirect()->back()
            ->with('success', 'Solicitud rechazada.');
    }
}

