<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Cliente;
use App\Models\Documento;
use App\Models\TipoDocumento;
use App\Models\HistorialCambio;
use App\Services\FileStorageService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class DocumentoController extends Controller
{
    protected $fileService;

    public function __construct(FileStorageService $fileService)
    {
        $this->fileService = $fileService;
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'cliente_id' => 'required|exists:clientes,id',
            'tipo_documento_id' => 'required|exists:tipos_documento,id',
            'archivo' => 'required|file|max:10240', // 10MB
            'fecha_vencimiento' => 'nullable|date',
            'observaciones' => 'nullable|string',
        ]);

        $cliente = Cliente::findOrFail($validated['cliente_id']);
        
        // Validar y guardar archivo
        $resultado = $this->fileService->guardarDocumento(
            $request->file('archivo'),
            $cliente->rif
        );

        if (!$resultado['valid']) {
            return back()->withErrors(['archivo' => $resultado['error']]);
        }

        // Verificar si ya existe un documento del mismo tipo
        $tipoDocumento = TipoDocumento::findOrFail($validated['tipo_documento_id']);
        $documentoExistente = Documento::where('cliente_id', $validated['cliente_id'])
            ->where('tipo_documento_id', $validated['tipo_documento_id'])
            ->first();

        if ($documentoExistente) {
            // Eliminar documento anterior
            Storage::disk('public')->delete($documentoExistente->ruta_archivo);
            $documentoExistente->delete();
            
            // Registrar en historial
            HistorialCambio::create([
                'cliente_id' => $validated['cliente_id'],
                'usuario_id' => auth()->id(),
                'accion_realizada' => "Documento '{$tipoDocumento->nombre}' reemplazado.",
            ]);
        }

        // Crear documento
        $documento = Documento::create([
            'cliente_id' => $validated['cliente_id'],
            'tipo_documento_id' => $validated['tipo_documento_id'],
            'nombre_archivo' => $resultado['nombre_archivo'],
            'ruta_archivo' => $resultado['ruta_archivo'],
            'fecha_vencimiento' => $validated['fecha_vencimiento'] ?? null,
            'observaciones' => $validated['observaciones'] ?? null,
            'creado_por_id' => auth()->id(),
        ]);

        // Registrar en historial si no fue reemplazo
        if (!$documentoExistente) {
            HistorialCambio::create([
                'cliente_id' => $validated['cliente_id'],
                'usuario_id' => auth()->id(),
                'accion_realizada' => "Documento '{$tipoDocumento->nombre}' subido.",
            ]);
        }

        return redirect()->back()
            ->with('success', $documentoExistente ? 'Documento reemplazado exitosamente.' : 'Documento subido exitosamente.');
    }

    public function destroy(Documento $documento)
    {
        $clienteId = $documento->cliente_id;
        $tipoDocumento = $documento->tipoDocumento;
        
        // Eliminar archivo físico
        Storage::disk('public')->delete($documento->ruta_archivo);
        
        $documento->delete();

        // Registrar en historial
        HistorialCambio::create([
            'cliente_id' => $clienteId,
            'usuario_id' => auth()->id(),
            'accion_realizada' => "Documento '{$tipoDocumento->nombre}' eliminado.",
        ]);

        return redirect()->back()
            ->with('success', 'Documento eliminado exitosamente.');
    }

    public function download(Documento $documento)
    {
        if (!Storage::disk('public')->exists($documento->ruta_archivo)) {
            abort(404, 'Archivo no encontrado');
        }

        return Storage::disk('public')->download(
            $documento->ruta_archivo,
            $documento->nombre_archivo
        );
    }
}

