<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Cliente;
use App\Models\Documento;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index()
    {
        // Total de clientes
        $totalClientes = Cliente::count();

        // Estadísticas por estatus
        $clientesPorEstatus = [
            'VENCIDO' => 0,
            'ALERTA' => 0,
            'SIN DOCUMENTOS' => 0,
            'ACTUALIZADO' => 0
        ];

        $clientes = Cliente::with('documentos')->get();
        foreach ($clientes as $cliente) {
            $estatus = $cliente->calcularEstatus();
            $clientesPorEstatus[$estatus['texto']] = ($clientesPorEstatus[$estatus['texto']] ?? 0) + 1;
        }

        // Documentos por tipo
        $documentosPorTipo = \App\Models\TipoDocumento::withCount('documentos')
            ->orderBy('documentos_count', 'desc')
            ->get();

        // Clientes por vendedor
        $clientesPorVendedor = \App\Models\User::vendedores()
            ->withCount('clientes')
            ->orderBy('clientes_count', 'desc')
            ->get();

        // Documentos vencidos vs próximos a vencer
        $documentosVencidos = Documento::whereNotNull('fecha_vencimiento')
            ->where('fecha_vencimiento', '<', now())
            ->count();
        
        $documentosProximos = Documento::whereNotNull('fecha_vencimiento')
            ->where('fecha_vencimiento', '>=', now())
            ->where('fecha_vencimiento', '<=', now()->addMonths(6))
            ->count();

        // Documentos vencidos o próximos a vencer (para lista)
        $documentosProblema = Documento::with(['cliente', 'tipoDocumento'])
            ->whereNotNull('fecha_vencimiento')
            ->where('fecha_vencimiento', '<=', now()->addMonths(6))
            ->orderBy('fecha_vencimiento')
            ->get();

        // Procesar documentos
        $listaVencidos = [];
        $listaAlerta = [];
        $clientesConProblemas = [];

        $hoy = now()->startOfDay();

        foreach ($documentosProblema as $doc) {
            $fechaVenc = \Carbon\Carbon::parse($doc->fecha_vencimiento)->startOfDay();
            $clienteId = $doc->cliente_id;

            if ($fechaVenc->lt($hoy)) {
                if (!isset($listaVencidos[$clienteId])) {
                    $listaVencidos[$clienteId] = [
                        'cliente' => $doc->cliente,
                        'documentos' => []
                    ];
                }
                $listaVencidos[$clienteId]['documentos'][] = $doc;
                $clientesConProblemas[$clienteId] = true;
            } else {
                if (!isset($listaAlerta[$clienteId])) {
                    $listaAlerta[$clienteId] = [
                        'cliente' => $doc->cliente,
                        'documentos' => []
                    ];
                }
                $listaAlerta[$clienteId]['documentos'][] = $doc;
                $clientesConProblemas[$clienteId] = true;
            }
        }

        // Actividades recientes (historial de cambios)
        $actividadesRecientes = \App\Models\HistorialCambio::with(['usuario', 'cliente'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return view('admin.dashboard', [
            'totalClientes' => $totalClientes,
            'clientesPorEstatus' => $clientesPorEstatus,
            'documentosPorTipo' => $documentosPorTipo,
            'clientesPorVendedor' => $clientesPorVendedor,
            'documentosVencidos' => $documentosVencidos,
            'documentosProximos' => $documentosProximos,
            'listaVencidos' => $listaVencidos,
            'listaAlerta' => $listaAlerta,
            'clientesConProblemas' => count($clientesConProblemas),
            'actividadesRecientes' => $actividadesRecientes,
        ]);
    }
}

