<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

class BackupController extends Controller
{
    public function index()
    {
        $backups = $this->obtenerBackups();
        
        return view('admin.backup.index', [
            'backups' => $backups,
        ]);
    }

    public function crearBackup()
    {
        try {
            $config = config('database.connections.mysql');
            $database = $config['database'];
            $username = $config['username'];
            $password = $config['password'];
            $host = $config['host'];
            $port = $config['port'] ?? 3306;

            // Crear directorio de backups si no existe
            $backupDir = storage_path('app/backups');
            if (!File::exists($backupDir)) {
                File::makeDirectory($backupDir, 0755, true);
            }

            // Nombre del archivo
            $filename = 'backup_' . $database . '_' . date('Y-m-d_His') . '.sql';
            $filepath = $backupDir . '/' . $filename;

            // Comando mysqldump
            $command = sprintf(
                'mysqldump --host=%s --port=%s --user=%s --password=%s %s > %s',
                escapeshellarg($host),
                escapeshellarg($port),
                escapeshellarg($username),
                escapeshellarg($password),
                escapeshellarg($database),
                escapeshellarg($filepath)
            );

            // Ejecutar comando
            exec($command, $output, $returnVar);

            if ($returnVar !== 0) {
                // Intentar sin password en la línea de comandos (más seguro)
                $command = sprintf(
                    'mysqldump --host=%s --port=%s --user=%s -p%s %s > %s',
                    escapeshellarg($host),
                    escapeshellarg($port),
                    escapeshellarg($username),
                    escapeshellarg($password),
                    escapeshellarg($database),
                    escapeshellarg($filepath)
                );
                
                // Usar variable de entorno para password
                putenv("MYSQL_PWD={$password}");
                exec($command, $output, $returnVar);
                putenv("MYSQL_PWD=");
            }

            if ($returnVar !== 0 || !File::exists($filepath)) {
                return redirect()->back()
                    ->with('error', 'Error al crear el backup. Verifica que mysqldump esté instalado y accesible.');
            }

            // Verificar que el archivo no esté vacío
            if (File::size($filepath) < 100) {
                File::delete($filepath);
                return redirect()->back()
                    ->with('error', 'El backup se creó pero está vacío. Verifica las credenciales de la base de datos.');
            }

            return redirect()->back()
                ->with('success', "Backup creado exitosamente: {$filename}");

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error al crear backup: ' . $e->getMessage());
        }
    }

    public function descargarBackup($filename)
    {
        $filepath = storage_path('app/backups/' . $filename);

        if (!File::exists($filepath)) {
            abort(404, 'Backup no encontrado');
        }

        return response()->download($filepath, $filename);
    }

    public function eliminarBackup($filename)
    {
        $filepath = storage_path('app/backups/' . $filename);

        if (!File::exists($filepath)) {
            return redirect()->back()
                ->with('error', 'Backup no encontrado');
        }

        File::delete($filepath);

        return redirect()->back()
            ->with('success', 'Backup eliminado exitosamente');
    }

    private function obtenerBackups()
    {
        $backupDir = storage_path('app/backups');
        
        if (!File::exists($backupDir)) {
            return [];
        }

        $files = File::files($backupDir);
        $backups = [];

        foreach ($files as $file) {
            if ($file->getExtension() === 'sql') {
                $backups[] = [
                    'nombre' => $file->getFilename(),
                    'tamaño' => $this->formatearTamaño($file->getSize()),
                    'fecha' => date('d/m/Y H:i:s', $file->getMTime()),
                    'timestamp' => $file->getMTime(),
                ];
            }
        }

        // Ordenar por fecha (más reciente primero)
        usort($backups, function($a, $b) {
            return $b['timestamp'] - $a['timestamp'];
        });

        return $backups;
    }

    private function formatearTamaño($bytes)
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= pow(1024, $pow);
        return round($bytes, 2) . ' ' . $units[$pow];
    }
}

