<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;

class GeneratePWAIcons extends Command
{
    protected $signature = 'pwa:generate-icons {--source= : Ruta a una imagen fuente (512x512px recomendado)}';
    protected $description = 'Genera iconos PWA desde una imagen fuente o crea placeholders';

    public function handle()
    {
        $iconsPath = public_path('vendedor/assets/icons');
        
        // Asegurar que el directorio existe
        if (!File::exists($iconsPath)) {
            File::makeDirectory($iconsPath, 0755, true);
        }
        
        $source = $this->option('source');
        
        if ($source && File::exists($source)) {
            $this->info('Generando iconos desde imagen fuente...');
            $this->generateFromImage($source, $iconsPath);
        } else {
            $this->info('Generando iconos placeholder...');
            $this->generatePlaceholders($iconsPath);
        }
        
        $this->info('');
        $this->info('✅ Iconos generados en: ' . $iconsPath);
        $this->info('');
        $this->warn('⚠️  Nota: Los iconos placeholder son temporales.');
        $this->warn('   Reemplázalos con tus iconos personalizados cuando estén listos.');
        $this->info('');
        
        return Command::SUCCESS;
    }
    
    private function generatePlaceholders($iconsPath)
    {
        $sizes = [
            ['name' => 'icon-16x16.png', 'size' => 16],
            ['name' => 'icon-32x32.png', 'size' => 32],
            ['name' => 'icon-72x72.png', 'size' => 72],
            ['name' => 'icon-96x96.png', 'size' => 96],
            ['name' => 'icon-128x128.png', 'size' => 128],
            ['name' => 'icon-144x144.png', 'size' => 144],
            ['name' => 'icon-152x152.png', 'size' => 152],
            ['name' => 'icon-192x192.png', 'size' => 192],
            ['name' => 'icon-384x384.png', 'size' => 384],
            ['name' => 'icon-512x512.png', 'size' => 512],
            ['name' => 'shortcut-clientes.png', 'size' => 96],
            ['name' => 'shortcut-alertas.png', 'size' => 96],
        ];
        
        foreach ($sizes as $icon) {
            $this->createPlaceholderIcon($iconsPath . '/' . $icon['name'], $icon['size']);
            $this->line('   ✓ ' . $icon['name']);
        }
    }
    
    private function createPlaceholderIcon($path, $size)
    {
        // Crear imagen usando GD si está disponible
        if (!extension_loaded('gd')) {
            // Si GD no está disponible, crear un SVG simple
            $this->createSVGIcon($path, $size);
            return;
        }
        
        $image = imagecreatetruecolor($size, $size);
        
        // Color de fondo (azul primario)
        $bgColor = imagecolorallocate($image, 13, 110, 253);
        imagefill($image, 0, 0, $bgColor);
        
        // Color del texto (blanco)
        $textColor = imagecolorallocate($image, 255, 255, 255);
        
        // Texto "PWA" o iniciales
        $text = $size >= 96 ? 'PWA' : 'P';
        $fontSize = max(12, $size / 8);
        
        // Calcular posición del texto (centrado)
        $bbox = imagettfbbox($fontSize, 0, $this->getFontPath(), $text);
        $textWidth = $bbox[4] - $bbox[0];
        $textHeight = $bbox[1] - $bbox[7];
        $x = ($size - $textWidth) / 2;
        $y = ($size + $textHeight) / 2;
        
        // Intentar usar fuente TrueType, si no está disponible usar fuente built-in
        if (function_exists('imagettftext') && $this->getFontPath()) {
            imagettftext($image, $fontSize, 0, $x, $y, $textColor, $this->getFontPath(), $text);
        } else {
            // Usar fuente built-in
            $font = 5; // Fuente grande
            $textWidth = imagefontwidth($font) * strlen($text);
            $textHeight = imagefontheight($font);
            $x = ($size - $textWidth) / 2;
            $y = ($size - $textHeight) / 2;
            imagestring($image, $font, $x, $y, $text, $textColor);
        }
        
        // Guardar como PNG
        imagepng($image, $path);
        imagedestroy($image);
    }
    
    private function createSVGIcon($path, $size)
    {
        $svg = <<<SVG
<svg width="{$size}" height="{$size}" xmlns="http://www.w3.org/2000/svg">
  <rect width="{$size}" height="{$size}" fill="#0d6efd"/>
  <text x="50%" y="50%" font-family="Arial, sans-serif" font-size="{$this->getSVGFontSize($size)}" 
        fill="white" text-anchor="middle" dominant-baseline="middle" font-weight="bold">PWA</text>
</svg>
SVG;
        File::put($path, $svg);
    }
    
    private function getSVGFontSize($size)
    {
        return max(12, $size / 6);
    }
    
    private function getFontPath()
    {
        // Intentar encontrar una fuente TrueType común
        $fonts = [
            'C:/Windows/Fonts/arial.ttf',
            'C:/Windows/Fonts/calibri.ttf',
            '/usr/share/fonts/truetype/dejavu/DejaVuSans-Bold.ttf',
        ];
        
        foreach ($fonts as $font) {
            if (file_exists($font)) {
                return $font;
            }
        }
        
        return null;
    }
    
    private function generateFromImage($source, $iconsPath)
    {
        if (!extension_loaded('gd')) {
            $this->error('La extensión GD no está disponible. No se pueden redimensionar imágenes.');
            $this->info('Usando iconos placeholder en su lugar...');
            $this->generatePlaceholders($iconsPath);
            return;
        }
        
        $sourceImage = $this->loadImage($source);
        if (!$sourceImage) {
            $this->error('No se pudo cargar la imagen fuente.');
            $this->info('Usando iconos placeholder en su lugar...');
            $this->generatePlaceholders($iconsPath);
            return;
        }
        
        $sizes = [
            ['name' => 'icon-16x16.png', 'size' => 16],
            ['name' => 'icon-32x32.png', 'size' => 32],
            ['name' => 'icon-72x72.png', 'size' => 72],
            ['name' => 'icon-96x96.png', 'size' => 96],
            ['name' => 'icon-128x128.png', 'size' => 128],
            ['name' => 'icon-144x144.png', 'size' => 144],
            ['name' => 'icon-152x152.png', 'size' => 152],
            ['name' => 'icon-192x192.png', 'size' => 192],
            ['name' => 'icon-384x384.png', 'size' => 384],
            ['name' => 'icon-512x512.png', 'size' => 512],
            ['name' => 'shortcut-clientes.png', 'size' => 96],
            ['name' => 'shortcut-alertas.png', 'size' => 96],
        ];
        
        foreach ($sizes as $icon) {
            $resized = imagecreatetruecolor($icon['size'], $icon['size']);
            imagecopyresampled(
                $resized, $sourceImage,
                0, 0, 0, 0,
                $icon['size'], $icon['size'],
                imagesx($sourceImage), imagesy($sourceImage)
            );
            imagepng($resized, $iconsPath . '/' . $icon['name']);
            imagedestroy($resized);
            $this->line('   ✓ ' . $icon['name']);
        }
        
        imagedestroy($sourceImage);
    }
    
    private function loadImage($path)
    {
        $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));
        
        switch ($extension) {
            case 'jpg':
            case 'jpeg':
                return imagecreatefromjpeg($path);
            case 'png':
                return imagecreatefrompng($path);
            case 'gif':
                return imagecreatefromgif($path);
            default:
                return null;
        }
    }
}

