<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;

class CheckPWAConfig extends Command
{
    protected $signature = 'pwa:check';
    protected $description = 'Verifica la configuración de la PWA';

    public function handle()
    {
        $this->info('Verificando configuración PWA...');
        $this->info('');
        
        $errors = 0;
        $warnings = 0;
        
        // Verificar claves VAPID
        $this->checkVAPIDKeys($errors, $warnings);
        
        // Verificar iconos
        $this->checkIcons($errors, $warnings);
        
        // Verificar Service Worker
        $this->checkServiceWorker($errors, $warnings);
        
        // Verificar Manifest
        $this->checkManifest($errors, $warnings);
        
        $this->info('');
        if ($errors === 0 && $warnings === 0) {
            $this->info('✅ ¡Configuración PWA completa y correcta!');
            return Command::SUCCESS;
        } elseif ($errors === 0) {
            $this->warn("⚠️  Configuración funcional con {$warnings} advertencia(s)");
            return Command::SUCCESS;
        } else {
            $this->error("❌ Se encontraron {$errors} error(es) y {$warnings} advertencia(s)");
            return Command::FAILURE;
        }
    }
    
    private function checkVAPIDKeys(&$errors, &$warnings)
    {
        $this->info('🔑 Verificando claves VAPID...');
        
        $publicKey = config('services.vapid.public_key');
        $privateKey = config('services.vapid.private_key');
        $subject = config('services.vapid.subject');
        
        if (empty($publicKey) || empty($privateKey)) {
            $this->error('   ❌ Claves VAPID no configuradas');
            $this->line('      Ejecuta: php artisan pwa:generate-vapid');
            $errors++;
        } else {
            $this->info('   ✅ Claves VAPID configuradas');
        }
        
        if (empty($subject) || $subject === 'mailto:admin@example.com') {
            $this->warn('   ⚠️  VAPID_SUBJECT no configurado o usa valor por defecto');
            $warnings++;
        } else {
            $this->info('   ✅ VAPID_SUBJECT configurado');
        }
    }
    
    private function checkIcons(&$errors, &$warnings)
    {
        $this->info('');
        $this->info('🎨 Verificando iconos PWA...');
        
        $iconsPath = public_path('vendedor/assets/icons');
        $requiredIcons = [
            'icon-192x192.png' => 'Requerido',
            'icon-512x512.png' => 'Requerido',
        ];
        
        $optionalIcons = [
            'icon-16x16.png',
            'icon-32x32.png',
            'icon-72x72.png',
            'icon-96x96.png',
            'icon-128x128.png',
            'icon-144x144.png',
            'icon-152x152.png',
            'icon-384x384.png',
            'shortcut-clientes.png',
            'shortcut-alertas.png',
        ];
        
        if (!File::exists($iconsPath)) {
            $this->error('   ❌ Directorio de iconos no existe: ' . $iconsPath);
            $errors++;
            return;
        }
        
        $missingRequired = [];
        foreach ($requiredIcons as $icon => $status) {
            if (!File::exists($iconsPath . '/' . $icon)) {
                $missingRequired[] = $icon;
            }
        }
        
        if (!empty($missingRequired)) {
            $this->error('   ❌ Iconos requeridos faltantes: ' . implode(', ', $missingRequired));
            $errors++;
        } else {
            $this->info('   ✅ Iconos requeridos presentes');
        }
        
        $missingOptional = [];
        foreach ($optionalIcons as $icon) {
            if (!File::exists($iconsPath . '/' . $icon)) {
                $missingOptional[] = $icon;
            }
        }
        
        if (!empty($missingOptional)) {
            $this->warn('   ⚠️  Iconos opcionales faltantes: ' . count($missingOptional));
            $warnings++;
        } else {
            $this->info('   ✅ Todos los iconos opcionales presentes');
        }
    }
    
    private function checkServiceWorker(&$errors, &$warnings)
    {
        $this->info('');
        $this->info('⚙️  Verificando Service Worker...');
        
        $swPath = public_path('vendedor/sw.js');
        
        if (!File::exists($swPath)) {
            $this->error('   ❌ Service Worker no encontrado: ' . $swPath);
            $errors++;
        } else {
            $this->info('   ✅ Service Worker encontrado');
            
            $content = File::get($swPath);
            if (strpos($content, 'CACHE_NAME') === false) {
                $this->warn('   ⚠️  Service Worker parece estar vacío o incompleto');
                $warnings++;
            }
        }
    }
    
    private function checkManifest(&$errors, &$warnings)
    {
        $this->info('');
        $this->info('📄 Verificando Manifest...');
        
        $manifestPath = public_path('vendedor/manifest.json');
        
        if (!File::exists($manifestPath)) {
            $this->error('   ❌ Manifest no encontrado: ' . $manifestPath);
            $errors++;
            return;
        }
        
        $this->info('   ✅ Manifest encontrado');
        
        $manifest = json_decode(File::get($manifestPath), true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->error('   ❌ Manifest tiene errores de JSON: ' . json_last_error_msg());
            $errors++;
            return;
        }
        
        $requiredFields = ['name', 'short_name', 'start_url', 'display', 'icons'];
        $missingFields = [];
        
        foreach ($requiredFields as $field) {
            if (!isset($manifest[$field]) || empty($manifest[$field])) {
                $missingFields[] = $field;
            }
        }
        
        if (!empty($missingFields)) {
            $this->error('   ❌ Campos requeridos faltantes: ' . implode(', ', $missingFields));
            $errors++;
        } else {
            $this->info('   ✅ Todos los campos requeridos presentes');
        }
        
        if (empty($manifest['icons']) || count($manifest['icons']) < 2) {
            $this->warn('   ⚠️  Manifest tiene menos de 2 iconos definidos');
            $warnings++;
        }
    }
}

